#include <unistd.h>

#include "c.h"
#include "postgres.h"
#include "fmgr.h"
#include "miscadmin.h"
#include "access/twophase.h"
#include "catalog/pg_control.h"
#include "postmaster/bgworker.h"
#include "postmaster/syslogger.h"
#include "storage/proc.h"
#include "utils/guc.h"

PG_MODULE_MAGIC;

extern void _PG_init(void);
extern void log_worker_main(Datum main_arg);

static int	log_stream_id = 0;
static int	msg_delay;

void
_PG_init(void)
{
	BackgroundWorker worker;
	LogStream	*log_stream;

	log_stream_id = get_log_stream("log_gen", &log_stream);

	/*
	 * Make the stream configurable via GUC.
	 */
	DefineCustomStringVariable(
		"log_gen.log_directory",

		"Sets the destination directory for log files.",
		"Can be specified as relative to the data directory "
		"or as absolute path.",
		&log_stream->directory,
		"log",
		PGC_SIGHUP,
		0,
		guc_check_canonical_path, NULL, NULL);

	DefineCustomStringVariable(
		"log_gen.log_filename",
		"File name the test messages are written to.",
		NULL,
		&log_stream->filename,
		"log_gen-%Y-%m-%d_%H%M%S.log",
		PGC_SIGHUP,
		0,
		NULL, NULL, NULL);

	DefineCustomIntVariable(
		"log_gen.log_file_mode",

		"Sets the file permissions for log files.",
		NULL,

		&log_stream->file_mode,
		0600, 0000, 0777,
		PGC_SIGHUP,
		0,
		NULL, NULL, guc_show_log_file_mode);

	DefineCustomIntVariable(
		"log_gen.log_rotation_age",

		"Automatic log file rotation will occur after N minutes.",
		NULL,

		&log_stream->rotation_age,
		HOURS_PER_DAY * MINS_PER_HOUR, 0, INT_MAX / SECS_PER_MINUTE,
		PGC_SIGHUP,
		GUC_UNIT_MIN,
		NULL, NULL, NULL);

	DefineCustomIntVariable(
		"log_gen.log_rotation_size",

		"Automatic log file rotation will occur after N kilobytes.",
		NULL,

		&log_stream->rotation_size,
		10 * 1024, 0, INT_MAX / 1024,
		PGC_SIGHUP,
		GUC_UNIT_KB,
		NULL, NULL, NULL);

	DefineCustomBoolVariable(
		"log_gen.log_truncate_on_rotation",

		"Truncate existing log files of same name during log rotation.",
		NULL,
		&log_stream->truncate_on_rotation,
		false,
		PGC_SIGHUP,
		0,
		NULL, NULL, NULL);

	DefineCustomStringVariable(
		"log_gen.log_line_prefix",

		"Controls information prefixed to each log line.",
		NULL,

		&log_stream->line_prefix,
		"",
		PGC_SIGHUP,
		0,
		NULL, NULL, NULL);
	/*
	 * Log settings end here.
	 */

	DefineCustomIntVariable(
		"log_gen.delay",
		"Sleep time in milliseconds of the log worker.",
		NULL,
		&msg_delay,
		1000, 1, INT_MAX,
		PGC_POSTMASTER,
		GUC_UNIT_MS,
		NULL, NULL, NULL);

	worker.bgw_flags = 0;
	worker.bgw_start_time = BgWorkerStart_RecoveryFinished;
	worker.bgw_restart_time = 0;
	sprintf(worker.bgw_name, "log worker");
	sprintf(worker.bgw_library_name, "log_gen");
	sprintf(worker.bgw_function_name, "log_worker_main");
	worker.bgw_main_arg = (Datum) 0;
	worker.bgw_notify_pid = 0;
	RegisterBackgroundWorker(&worker);
}

static volatile sig_atomic_t got_sigterm = false;

static void
log_worker_sigterm(SIGNAL_ARGS)
{
	int			save_errno = errno;

	got_sigterm = true;
	SetLatch(MyLatch);

	errno = save_errno;
}

void
log_worker_main(Datum main_arg)
{
	pqsignal(SIGTERM, log_worker_sigterm);
	BackgroundWorkerUnblockSignals();

	for (;;)
	{
		int	rc;

		rc = WaitLatch(MyLatch, WL_LATCH_SET | WL_POSTMASTER_DEATH |
					   WL_TIMEOUT, msg_delay, 0);
		if (rc & WL_POSTMASTER_DEATH)
			break;

		if (got_sigterm)
			break;

		ereport(LOG, (errmsg("Just another useless log message."),
					  errstream(log_stream_id)));

		ResetLatch(MyLatch);
	}

}
